/*
* eeectl
* Copyright (C) 2008 Anthony A Z <dci@cpp.in>
* http://www.cpp.in/dev/eeectl/
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License along
* with this program; if not, write to the Free Software Foundation, Inc.,
* 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#include <windows.h>
#include <wchar.h>

#include "runtime.h"
#include "table.h"
#include "txticon.h"

#include "eeehw.h"

#include "eeectl.h"
#include "ss_backlight.h"
#include "ss_fan.h"
#include "ss_speed.h"
#include "ss_temp.h"



#define WNDSTRING  L"dcieeectl"

#define WM_TRAYICON ( WM_APP + 0x10 )
#define WM_EXECMCMD ( WM_APP + 0x11 )


#define TIMER_ICON 1
#define TIMER_SAVE 2
#define TIMER_TIME 3


#define QUIT_NORMAL  0
#define QUIT_ERROR   1
#define QUIT_RESTART 10


#define CFLEN 2048

#define SAVE_COL_NAME 0
#define SAVE_COL_NUM  1

#define MENU_COL_NAME 0
#define MENU_COL_MODE 1
#define MENU_COL_NUM  2

#define ICON_COL_NAME  0
#define ICON_COL_COLOR 1
#define ICON_COL_NUM   2

#define KEYS_COL_KEYS 0
#define KEYS_COL_CALL 1
#define KEYS_COL_NUM  2



/* globals */
int     iExitCode = QUIT_NORMAL;
wchar_t szConfigFile [ MAX_PATH ];
HWND    hMainWnd;


sth_t core_save;
sth_t core_menu;
sth_t core_icon;
sth_t core_keys;

UINT  core_icon_delay;
UINT  core_icon_current;


/* forward decls */
extern eeectl_loader_t eeectl_loader;
extern eeectl_subsys_t eeectl_subsys_core;
extern eeectl_subsys_t * eeectl_subsystems [ ];
eeectl_subsys_t * ss_lookup ( wchar_t * name );


/********************************************************/
/***************** EEECTL LOADER OBJECT *****************/
/********************************************************/
UINT loader_setup_read ( eeectl_subsys_t * subsys, wchar_t * name, wchar_t * buf, UINT num )
{
	wchar_t pname [ 128 ];
	wsprintf ( pname, L"%s.%s", subsys->name, name );
	return GetPrivateProfileString ( L"settings", pname, L"", buf, num, szConfigFile );
}

UINT loader_setup_write ( eeectl_subsys_t * subsys, wchar_t * name, wchar_t * buf )
{
	wchar_t pname [ 128 ];
	wsprintf ( pname, L"%s.%s", subsys->name, name );
	return WritePrivateProfileString ( L"settings", pname, buf, szConfigFile );
}

UINT loader_state_read ( eeectl_subsys_t * subsys, wchar_t * name, wchar_t * buf, UINT num )
{
	UINT i;
	for ( i = 0; i < st_height ( core_save ); ++i )
	{
		if ( 0 == mwcsicmp ( subsys->name, st_value_get ( core_save, i, SAVE_COL_NAME ) ) )
		{
			wchar_t pname [ 128 ];
			wsprintf ( pname, L"%s.%s", subsys->name, name );
			return GetPrivateProfileString ( L"States", pname, L"", buf, num, szConfigFile );
		}
	}
	* buf = 0;
	return 0;
}

UINT loader_state_write ( eeectl_subsys_t * subsys, wchar_t * name, wchar_t * buf )
{
	UINT i;
	for ( i = 0; i < st_height ( core_save ); ++i )
	{
		if ( 0 == mwcsicmp ( subsys->name, st_value_get ( core_save, i, SAVE_COL_NAME ) ) )
		{
			wchar_t pname [ 128 ];
			wsprintf ( pname, L"%s.%s", subsys->name, name );
			return WritePrivateProfileString ( L"states", pname, buf, szConfigFile );
		}
	}
	return 0;
}

void loader_ntf_icon_upd ( eeectl_subsys_t * subsys, wchar_t * name )
{
	UINT i;
	if ( NULL == name ) name = L"";
	/* */
	for ( i = 0; i < st_height ( core_icon ); ++i )
	{
		wchar_t * sname = mwcsdup ( st_value_get ( core_icon, i, ICON_COL_NAME ) );
		wchar_t * iname = mwcschr ( sname, '.' ); * iname = 0; ++iname;
		if ( ( 0 == mwcsicmp ( subsys->name, sname ) ) && ( ( 0 == ( * name ) ) || ( 0 == mwcsicmp ( name, iname ) ) ) )
		{
			mfree ( sname );
			core_icon_current = i - 1;
			SendMessage ( hMainWnd, WM_TIMER, TIMER_ICON, 0 );
			break;
		}
		else mfree ( sname );
	}
}

void loader_debug_msg ( eeectl_subsys_t * subsys, unsigned long level, wchar_t * fmt,... )
{
	if ( MSG_WARN <= level )
	{
		va_list va;
		wchar_t * buf = mmalloc ( 4096 );
		wchar_t title [ 64 ];
		va_start ( va, fmt );
		wvsprintf ( buf, fmt, va );
		va_end ( va );
		wsprintf ( title, L"eeectl::%s", subsys->name );
		MessageBox ( NULL, buf, title, MB_ICONEXCLAMATION | MB_TOPMOST );
		mfree ( buf );
	}
}

eeectl_loader_t eeectl_loader = { loader_setup_read, loader_setup_write, loader_state_read, loader_state_write, loader_ntf_icon_upd, loader_debug_msg };


/********************************************************/
/**************** "CORE" SUBSYSTEM STUB *****************/
/********************************************************/
BOOL core_init ( eeectl_loader_t * core )
{
	UINT i;
	wchar_t * confstring = ( wchar_t * ) mmalloc ( CFLEN * sizeof ( wchar_t ) );
	/* read core.save */
	core->setup_read ( &eeectl_subsys_core, L"Save", confstring, CFLEN );
	core_save = st_create ( SAVE_COL_NUM );
	st_serial_read ( core_save, confstring );
	/* check subsystems */
	for ( i = 0; i < st_height ( core_save ); ++i )
	{
		wchar_t * sname = st_value_get ( core_save, i, SAVE_COL_NAME );
		if ( NULL == ss_lookup ( sname ) )
		{
			core->pass_msg ( &eeectl_subsys_core, MSG_WARN, L"Unknown subsystem in Core.Save: '%s'.", sname );
			st_row_del ( core_save, i );
			--i;
		}
	}
	/* read core.menu */
	core->setup_read ( &eeectl_subsys_core, L"Menu", confstring, CFLEN );
	core_menu = st_create ( MENU_COL_NUM );
	st_serial_read ( core_menu, confstring );
	/* check subsystems */
	for ( i = 0; i < st_height ( core_menu ); ++i )
	{
		wchar_t * sname = st_value_get ( core_menu, i, MENU_COL_NAME );
		if ( NULL == ss_lookup ( sname ) )
		{
			core->pass_msg ( &eeectl_subsys_core, MSG_WARN, L"Unknown subsystem in Core.Menu: '%s'.", sname );
			st_row_del ( core_menu, i );
			--i;
		}
	}
	if ( 0 == st_height ( core_menu ) )
	{
		st_row_add ( core_menu );
		st_value_set ( core_menu, 0, MENU_COL_NAME, L"Core" );
		st_value_set ( core_menu, 0, MENU_COL_MODE, L"embed" );
	}
	/* read core.icon */
	core->setup_read ( &eeectl_subsys_core, L"Icon", confstring, CFLEN );
	core_icon = st_create ( ICON_COL_NUM );
	st_serial_read ( core_icon, confstring );
	/* check subsystems */
	for ( i = 0; i < st_height ( core_icon ); ++i )
	{
		wchar_t * sname = mwcsdup ( st_value_get ( core_icon, i, ICON_COL_NAME ) );
		wchar_t * dot   = mwcschr ( sname, '.' );
		if ( dot ) * dot = 0;
		if ( ( NULL == dot ) || ( NULL == ss_lookup ( sname ) ) )
		{
			core->pass_msg ( &eeectl_subsys_core, MSG_WARN, L"Invalid token in Core.Icon: '%s'.", sname );
			st_row_del ( core_icon, i );
			--i;
		}
		mfree ( sname );
	}
	/* read core.icondelay */
	core->setup_read ( &eeectl_subsys_core, L"IconDelay", confstring, CFLEN );
	core_icon_delay = mwtoi ( confstring );
	/* */
	core_icon_current = 0;
	/* read core.keys */
	core->setup_read ( &eeectl_subsys_core, L"Keys", confstring, CFLEN );
	core_keys = st_create ( KEYS_COL_NUM );
	st_serial_read ( core_keys, confstring );
	/* check subsystems */
	for ( i = 0; i < st_height ( core_keys ); ++i )
	{
		wchar_t * sname = mwcsdup ( st_value_get ( core_keys, i, KEYS_COL_CALL ) );
		wchar_t * dot   = mwcschr ( sname, '.' );
		if ( dot ) * dot = 0;
		if ( ( NULL == dot ) || ( NULL == ss_lookup ( sname ) ) )
		{
			core->pass_msg ( &eeectl_subsys_core, MSG_WARN, L"Invalid token in Core.Keys: '%s'.", sname );
			st_row_del ( core_keys, i );
			--i;
		}
		mfree ( sname );
	}
	/* */
	mfree ( confstring );
	/* */
	return TRUE;
}

void core_null ( ) { }

void core_stop ( BOOL restart )
{
	if ( core_keys ) st_destroy ( core_keys );
	if ( core_icon ) st_destroy ( core_icon );
	if ( core_menu ) st_destroy ( core_menu );
	if ( core_save ) st_destroy ( core_save );
	core_keys = NULL;
	core_icon = NULL;
	core_menu = NULL;
	core_save = NULL;
	core_icon_delay = 0;
}

UINT core_icontxt ( wchar_t * icon, wchar_t * txt, size_t max ) { * txt = 0; return 1; }

UINT core_menucrt ( HMENU menu, UINT base )
{
	AppendMenu ( menu, MF_BYCOMMAND | MF_STRING, base + 0, L"About" );
	AppendMenu ( menu, MF_BYCOMMAND | MF_STRING, base + 1, L"Exit" );
	return base + 2;
}

BOOL core_menuclk ( UINT id )
{
	if ( 0 == id )
	{
		MessageBox ( hMainWnd, L"eeectl v0.2.4 is  2008 dci.\n\ndci@cpp.in\nhttp://cpp.in/dev/eeectl/", L"About eeectl", MB_ICONINFORMATION | MB_TOPMOST );
	}
	else if ( 1 == id )
	{
		DestroyWindow ( hMainWnd );
	}
	return FALSE;
}

BOOL core_ntfevnt ( wchar_t * evname )
{
	if ( 0 == mwcsicmp ( L"_quit", evname ) )
	{
		iExitCode = QUIT_NORMAL;
		DestroyWindow ( hMainWnd );
	}
	else
	{
		HANDLE hTest = CreateFile ( evname, GENERIC_READ, FILE_SHARE_READ | FILE_SHARE_WRITE, NULL, OPEN_EXISTING, 0, NULL );
		if ( INVALID_HANDLE_VALUE != hTest )
		{
			CloseHandle ( hTest );
			mwcscpy ( szConfigFile, evname );
			iExitCode = QUIT_RESTART;
			DestroyWindow ( hMainWnd );
		}
	}
	return FALSE;
}

eeectl_subsys_t eeectl_subsys_core = { L"Core", L"eeectl core module", core_init, core_null, core_stop, core_null, core_null, core_icontxt, core_menucrt, core_menuclk, core_ntfevnt, core_null };


/********************************************************/
/******************* LOADER/OS STUFF ********************/
/********************************************************/
eeectl_subsys_t * eeectl_subsystems [ ] = { &eeectl_subsys_core, &eeectl_subsys_speed, &eeectl_subsys_fan, &eeectl_subsys_temp, &eeectl_subsys_backlight };

eeectl_subsys_t * ss_lookup ( wchar_t * name )
{
	UINT i;
	for ( i = 0; i < sizeof ( eeectl_subsystems ) / sizeof ( * eeectl_subsystems ); ++i )
	{
		if ( 0 == mwcsicmp ( name, eeectl_subsystems [ i ]->name ) ) return eeectl_subsystems [ i ];
	}
	return NULL;
}

LRESULT CALLBACK NotifyWndProc ( HWND hWnd, UINT nMsg, WPARAM wParam, LPARAM lParam )
{
	static NOTIFYICONDATA nid;
	/* */
	if ( WM_CREATE == nMsg )
	{
		UINT i;
		BOOL bStartOK = TRUE;
		/* init globals */
		hMainWnd = hWnd;
		/* start txticon */
		TIStart ( ( ( CREATESTRUCT * ) lParam )->hInstance );
		/* create icon */
		mmemset ( &nid, 0, sizeof ( nid ) );
		nid.cbSize = sizeof ( nid );
		nid.hIcon = TICreate ( L"", 0 );
		mwcscpy ( nid.szTip, L"eeectl" );
		nid.hWnd = hWnd;
		nid.uFlags = NIF_MESSAGE | NIF_ICON | NIF_TIP;
		nid.uCallbackMessage = WM_TRAYICON;
		Shell_NotifyIcon ( NIM_ADD, &nid );
		/* initialize subsystems */
		for ( i = 0; i < ( sizeof ( eeectl_subsystems ) / sizeof ( * eeectl_subsystems ) ); ++i )
		{
			if ( FALSE == eeectl_subsystems [ i ]->init ( &eeectl_loader ) )
			{
				wchar_t eb [ 256 ];
				wsprintf ( eb, L"Module '%s'(%s) refused to start", eeectl_subsystems [ i ]->name, eeectl_subsystems [ i ]->desc );
				MessageBox ( hWnd, eb, L"eeectl", MB_TOPMOST | MB_ICONSTOP );
				bStartOK = FALSE;
				break;
			}
		}
		/* register hotkeys */
		for ( i = 0; i < st_height ( core_keys ); ++i )
		{
			wchar_t * km = mwcsdup ( st_value_get ( core_keys, i, KEYS_COL_KEYS ) );
			wchar_t * kk = mwcschr ( km, '+' );
			if ( kk )
			{
				UINT mod = 0;
				UINT key = 0;
				/* */
				* kk = 0; ++kk;
				/* parse mod */
				while ( * km )
				{
					if ( 'a' == mlwchar ( * km ) ) mod |= MOD_ALT;
					else if ( 'c' == mlwchar ( * km ) ) mod |= MOD_CONTROL;
					else if ( 's' == mlwchar ( * km ) ) mod |= MOD_SHIFT;
					else if ( 'w' == mlwchar ( * km ) ) mod |= MOD_WIN;
					++km;
				}
				/* parse key */
				if ( ( 'f' == mlwchar ( * kk ) ) && IS_NUMERIC ( * ( kk + 1 ) ) ) key = VK_F1 + mwtoi ( kk + 1 ) - 1;
				else if ( 0 == mwcsicmp ( L"esc", kk ) ) key = VK_ESCAPE;
				else if ( 0 == mwcsicmp ( L"space", kk ) ) key = VK_SPACE;
				else if ( 0 == mwcsicmp ( L"left", kk ) ) key = VK_LEFT;
				else if ( 0 == mwcsicmp ( L"up", kk ) ) key = VK_UP;
				else if ( 0 == mwcsicmp ( L"right", kk ) ) key = VK_RIGHT;
				else if ( 0 == mwcsicmp ( L"down", kk ) ) key = VK_DOWN;
				else if ( 0 == mwcsicmp ( L"ins", kk ) ) key = VK_INSERT;
				else if ( 0 == mwcsicmp ( L"del", kk ) ) key = VK_DELETE;
				else if ( 0 == mwcsicmp ( L"home", kk ) ) key = VK_HOME;
				else if ( 0 == mwcsicmp ( L"end", kk ) ) key = VK_END;
				else if ( '%' == * kk ) key = mwcstoul ( kk + 1, NULL, 16 );
				else key = muwchar ( * kk );
				/* */
				if ( key ) RegisterHotKey ( hWnd, i, mod, key );
			}
			/* */
			mfree ( km );
		}
		/* schedule events */
		SetTimer ( hWnd, TIMER_TIME, 50, NULL );
		SetTimer ( hWnd, TIMER_ICON, 500, NULL );
		/* */
		if ( bStartOK )
		{
			iExitCode = QUIT_NORMAL;
			return 0;
		}
		else
		{
			iExitCode = QUIT_ERROR;
			return ( -1 );
		}
	}
	else if ( WM_DESTROY == nMsg )
	{
		UINT i;
		/* unregister hotkeys */
		for ( i = 0; i < st_height ( core_keys ); ++i ) UnregisterHotKey ( hWnd, i );
		/* stop subsystems */
		for ( i = ( sizeof ( eeectl_subsystems ) / sizeof ( * eeectl_subsystems ) ); i > 0; --i )
		{
			eeectl_subsystems [ i - 1 ]->save ( );
			eeectl_subsystems [ i - 1 ]->stop ( QUIT_RESTART == iExitCode );
		}
		/* free icon stuff */
		KillTimer ( hWnd, TIMER_ICON );
		Shell_NotifyIcon ( NIM_DELETE, &nid );
		TIDestroy ( nid.hIcon );
		TIStop ( );
		/* quit */
		PostQuitMessage ( iExitCode );
		return 0;
	}
	else if ( WM_POWERBROADCAST == nMsg )
	{
		UINT i;
		static BOOL suspended = FALSE;
		if ( ( FALSE == suspended ) && ( ( PBT_APMQUERYSUSPEND == wParam ) || ( PBT_APMSUSPEND == wParam ) ) )
		{
			suspended = TRUE;
			for ( i = 0; i < ( sizeof ( eeectl_subsystems ) / sizeof ( * eeectl_subsystems ) ); ++i ) eeectl_subsystems [ i ]->suspend ( );
			return TRUE;
		}
		else if ( suspended && ( ( PBT_APMRESUMESUSPEND == wParam ) || ( PBT_APMRESUMEAUTOMATIC == wParam ) ) )
		{
			suspended = FALSE;
			for ( i = 0; i < ( sizeof ( eeectl_subsystems ) / sizeof ( * eeectl_subsystems ) ); ++i ) eeectl_subsystems [ i ]->resume ( );
			return 0;
		}
	}
	else if ( WM_QUERYENDSESSION == nMsg )
	{
		UINT i;
		HMODULE hUser = GetModuleHandle ( L"USER32.DLL" );
		FARPROC fpShutdownBlockReasonCreate  = GetProcAddress ( hUser, "ShutdownBlockReasonCreate" );
		FARPROC fpShutdownBlockReasonDestroy = GetProcAddress ( hUser, "ShutdownBlockReasonDestroy" );
		if ( fpShutdownBlockReasonCreate ) ( ( BOOL ( WINAPI * ) ( HWND, LPCWSTR ) ) fpShutdownBlockReasonCreate ) ( hWnd, L"Shutdown is in progress, please wait..." );
		for ( i = 0; i < ( sizeof ( eeectl_subsystems ) / sizeof ( * eeectl_subsystems ) ); ++i ) eeectl_subsystems [ i ]->suspend ( );
		if ( fpShutdownBlockReasonDestroy ) ( ( BOOL ( WINAPI * ) ( HWND ) ) fpShutdownBlockReasonDestroy ) ( hWnd );
		return TRUE;
	}


	else if ( WM_TIMER == nMsg )
	{
		if ( TIMER_ICON == wParam )
		{
			KillTimer ( hWnd, TIMER_ICON );
			/* */
			if ( st_height ( core_icon ) )
			{
				/* wow that looks bad */
				static BOOL not_first_call = FALSE;
				if ( FALSE == not_first_call ) not_first_call = TRUE;
				else ++core_icon_current;
				if ( st_height ( core_icon ) <= core_icon_current ) core_icon_current = 0;
				/* this totally needs to be rewritten */
				if ( 1 )
				{
					eeectl_subsys_t * ss;
					wchar_t * sname = mwcsdup ( st_value_get ( core_icon, core_icon_current, ICON_COL_NAME ) );
					wchar_t * iname = mwcschr ( sname, '.' ); * iname = 0; ++iname;
					if ( ss = ss_lookup ( sname ) )
					{
						wchar_t * colorval;
						wchar_t textbuf [ 128 ];
						HICON old;
						/* */
						* textbuf = 0;
						old = nid.hIcon;
						ss->icontxt ( iname, textbuf, sizeof ( textbuf ) / sizeof ( * textbuf ) );
						colorval = st_value_get ( core_icon, core_icon_current, ICON_COL_COLOR );
						nid.hIcon = TICreate ( textbuf, ( * colorval ) ? mwcstoul ( colorval, NULL, 16 ) : RGB( 0, 255, 255 ) );
						Shell_NotifyIcon ( NIM_MODIFY, &nid );
						TIDestroy ( old );
						/* restart timer */
						if ( core_icon_delay ) SetTimer ( hWnd, TIMER_ICON, core_icon_delay, NULL );
					}
				}
			}
		}
		else if ( TIMER_SAVE == wParam )
		{
			UINT i;
			KillTimer ( hWnd, TIMER_SAVE );
			for ( i = 0; i < ( sizeof ( eeectl_subsystems ) / sizeof ( * eeectl_subsystems ) ); ++i ) eeectl_subsystems [ i ]->save ( );
		}
		else if ( TIMER_TIME == wParam )
		{
			UINT i;
			KillTimer ( hWnd, TIMER_TIME );
			for ( i = 0; i < ( sizeof ( eeectl_subsystems ) / sizeof ( * eeectl_subsystems ) ); ++i ) eeectl_subsystems [ i ]->ntftime ( );
			SetTimer ( hWnd, TIMER_TIME, 5000, NULL );
		}
	}
	else if ( WM_HOTKEY == nMsg )
	{
		wchar_t * call = mwcsdup ( st_value_get ( core_keys, ( unsigned long ) wParam, KEYS_COL_CALL ) );
		wchar_t * dot  = mwcschr ( call, '.' );
		if ( dot )
		{
			eeectl_subsys_t * ss;
			* dot = 0;
			if ( ss = ss_lookup ( call ) )
				if ( ss->ntfevnt ( dot + 1 ) ) SetTimer ( hWnd, TIMER_SAVE, 5000, NULL );
		}
		mfree ( call );
	}
	else if ( WM_TRAYICON == nMsg )
	{
		if ( WM_LBUTTONDOWN == lParam )
		{
			/* reschedule icon change */
			SetTimer ( hWnd, TIMER_ICON, 10, NULL );
		}
		else if ( WM_RBUTTONUP == lParam )
		{
			UINT i, nSubMenus = 0, miID = 1;
			HANDLE * hSubMenus = ( HANDLE ) mmalloc ( st_height ( core_menu ) * sizeof ( HANDLE ) );
			UINT   * uIDBegin  = ( UINT * ) mmalloc ( st_height ( core_menu ) * sizeof ( UINT ) );
			UINT   * uIDEnd    = ( UINT * ) mmalloc ( st_height ( core_menu ) * sizeof ( UINT ) );
			/* create menu */
			HMENU hRootMenu = CreatePopupMenu ( );
			/* fill menu */
			for ( i = 0; i < st_height ( core_menu ); ++i )
			{
				eeectl_subsys_t * subsys = ss_lookup ( st_value_get ( core_menu, i, MENU_COL_NAME ) );
				uIDBegin [ i ] = miID;
				/* */
				if ( 0 == mwcsicmp ( L"popup", st_value_get ( core_menu, i, MENU_COL_MODE ) ) )
				{
					UINT add;
					hSubMenus [ nSubMenus ] = CreatePopupMenu ( );
					if ( add = subsys->menucrt ( hSubMenus [ nSubMenus ], miID ) )
					{
						miID += add;
						AppendMenu ( hRootMenu, MF_STRING | MF_POPUP, ( UINT_PTR ) hSubMenus [ nSubMenus ], subsys->name ); 
						++nSubMenus;
					}
					else
					{
						DestroyMenu ( hSubMenus [ nSubMenus ] );
					}
				}
				else
				{
					UINT add = subsys->menucrt ( hRootMenu, miID );
					if ( add )
					{
						miID += add;
					}
				}
				/* nothing added? */
				if ( miID == uIDBegin [ i ] )
				{
					wchar_t buf [ 128 ];
					wsprintf ( buf, L"(%s)", subsys->name );
					AppendMenu ( hRootMenu, MF_STRING | MF_GRAYED, 0, buf );
				}
				/* add sep */
				if ( i < ( st_height ( core_menu ) - 1 ) ) AppendMenu ( hRootMenu, MF_SEPARATOR, 0, NULL );
				/* */
				uIDEnd [ i ] = miID;
			}
			/* is there anything to show? */
			if ( 1 != miID )
			{
				POINT pCursor;
				/* show */
				GetCursorPos ( &pCursor );
				SetForegroundWindow ( hWnd );
				SetActiveWindow  ( hWnd );
				miID = TrackPopupMenuEx ( hRootMenu, TPM_RIGHTALIGN | TPM_BOTTOMALIGN | TPM_RETURNCMD, pCursor.x, pCursor.y, hWnd, NULL );
				/* dispatch result */
				if ( miID )
				{
					for ( i = 0; i < st_height ( core_menu ); ++i )
					{
						if ( ( miID >= uIDBegin [ i ] ) && ( miID < uIDEnd [ i ] ) )
						{
							if ( ss_lookup ( st_value_get ( core_menu, i, MENU_COL_NAME ) )->menuclk ( miID - uIDBegin [ i ] ) ) SetTimer ( hWnd, TIMER_SAVE, 5000, NULL );
							break;
						}
					}
				}
			}
			else MessageBox ( hWnd, L"There is nothing to show!", L"dci", MB_TOPMOST | MB_ICONEXCLAMATION );
			/* free */
			for ( i = 0; i < nSubMenus; ++i ) DestroyMenu ( hSubMenus [ i ] );
			DestroyMenu ( hRootMenu );
			mfree ( hSubMenus );
			mfree ( uIDBegin );
			mfree ( uIDEnd );
		}
		return 0;
	}
	else if ( WM_EXECMCMD == nMsg )
	{
		if ( wParam )
		{
			wchar_t * call = mwcsdup ( ( wchar_t * ) wParam );
			wchar_t * dot  = mwcschr ( call, '.' );
			if ( dot )
			{
				eeectl_subsys_t * ss;
				* dot = 0;
				if ( ss = ss_lookup ( call ) )
					if ( ss->ntfevnt ( dot + 1 ) ) SetTimer ( hWnd, TIMER_SAVE, 5000, NULL );
			}
			mfree ( call );
		}
	}
	/* */
	return DefWindowProc ( hWnd, nMsg, wParam, lParam );
}

/* app entry point */
int WINAPI wWinMain ( HINSTANCE hInst, HINSTANCE hPrevInst, LPWSTR lpCmdLine, int nCmdShow )
{
	int  ret = 0;
	HWND hRunning;
	/* read cmdline */
	sth_t cmdline    = st_create ( 2 );
	BOOL bSecondCell = FALSE;
	while ( lpCmdLine && * lpCmdLine )
	{
		BOOL bQuoteOpened = FALSE;
		wchar_t * tbptr   = lpCmdLine;
		wchar_t * teptr   = tbptr;
		while ( 1 )
		{
			if ( ( 0 == * teptr ) || ( ( 0x20 >= * teptr ) && ( FALSE == bQuoteOpened ) ) ) break;
			if ( '"' == * teptr )
			{
				if ( FALSE == bQuoteOpened )
				{
					if ( teptr == tbptr ) ++tbptr;
					bQuoteOpened = TRUE;
				}
				else
				{
					if ( * ( teptr + 1 ) <= 0x20 ) * teptr = 0;
					bQuoteOpened = FALSE;
				}
			}
			++teptr;
		}
		/* */
		if ( * teptr )
		{
			* teptr = 0;
			lpCmdLine = teptr + 1;
		}
		else lpCmdLine = teptr;
		while ( ( * lpCmdLine ) && ( 0x20 >= * lpCmdLine ) ) ++lpCmdLine;
		/* */
		if ( bSecondCell )
		{
			st_value_set ( cmdline, st_height ( cmdline ) - 1, 1, tbptr );
			bSecondCell = FALSE;
		}
		else
		{
			st_value_set ( cmdline, st_row_add ( cmdline ), 0, tbptr );
			bSecondCell = TRUE;
		}
	}
	/* try to find running app */
	if ( hRunning = FindWindow ( WNDSTRING, WNDSTRING ) )
	{
		/* have commands to send? */
		if ( st_height ( cmdline ) )
		{
			/* find process */
			DWORD dwProc = 0;
			GetWindowThreadProcessId ( hRunning, &dwProc );
			if ( dwProc )
			{
				/* open process */
				HANDLE hProc = OpenProcess ( PROCESS_VM_OPERATION | PROCESS_VM_READ | PROCESS_VM_WRITE, FALSE, dwProc );
				if ( hProc )
				{
					/* allocate remote buffer */
					VOID * remem = VirtualAllocEx ( hProc, NULL, 4096, MEM_COMMIT, PAGE_READWRITE );
					if ( remem )
					{
						UINT  i;
						DWORD dummy;
						for ( i = 0; i < st_height ( cmdline ); ++i )
						{
							if ( 0 == mwcsicmp ( st_value_get ( cmdline, i, 0 ), L"-c" ) )
							{
								wchar_t * config = st_value_get ( cmdline, i, 1 );
								WriteProcessMemory ( hProc, remem, L"core.", 10, &dummy );
								WriteProcessMemory ( hProc, ( ( BYTE * ) remem ) + 10, config, ( mwcslen ( config ) + 1 ) * sizeof ( * config ), &dummy );
								SendMessage ( hRunning, WM_EXECMCMD, ( WPARAM ) remem, 0 );
								break;
							}
						}
						/* */
						for ( i = 0; i < st_height ( cmdline ); ++i )
						{
							if ( 0 == mwcsicmp ( st_value_get ( cmdline, i, 0 ), L"-e" ) )
							{
								DWORD dummy;
								wchar_t * command = st_value_get ( cmdline, i, 1 );
								WriteProcessMemory ( hProc, remem, command, ( mwcslen ( command ) + 1 ) * sizeof ( * command ), &dummy );
								SendMessage ( hRunning, WM_EXECMCMD, ( WPARAM ) remem, 0 );
							}
						}
						/* */
						VirtualFreeEx ( hProc, remem, 4096, MEM_RELEASE );
					}
					else MessageBox ( NULL, L"Unable to alloc remote memory.", L"dci", MB_ICONSTOP | MB_TOPMOST );
					/* */
					CloseHandle ( hProc );
				}
				else MessageBox ( NULL, L"Unable to open process.", L"dci", MB_ICONSTOP | MB_TOPMOST );
			}
			else MessageBox ( NULL, L"Unable to read process ID.", L"dci", MB_ICONSTOP | MB_TOPMOST );
		}
		else MessageBox ( NULL, L"eeectl is already started.", L"dci", MB_ICONSTOP | MB_TOPMOST );
	}
	else
	{
		UINT i;
		HANDLE test;
		/* create config path */
		for ( i = 0; i < st_height ( cmdline ); ++i )
		{
			if ( 0 == mwcsicmp ( L"-c", st_value_get ( cmdline, i, 0 ) ) )
			{
				mwcscpy ( szConfigFile, st_value_get ( cmdline, i, 1 ) );
				break;
			}
		}
		if ( 0 == * szConfigFile )
		{
			GetModuleFileName ( NULL, szConfigFile, MAX_PATH );
			mwcscpy ( mwcsrchr ( szConfigFile, '.' ), L".ini" );
		}
		/* test if config exists */
		test = CreateFile ( szConfigFile, GENERIC_READ, FILE_SHARE_READ | FILE_SHARE_WRITE, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL );
		if ( INVALID_HANDLE_VALUE != test ) CloseHandle ( test );
		/* config exists? */
		if ( INVALID_HANDLE_VALUE == test )
		{
			wchar_t msgbuf [ MAX_PATH + 64 ];
			wsprintf ( msgbuf, L"Can't open config file '%s'.", szConfigFile );
			MessageBox ( NULL, msgbuf, L"dci", MB_ICONSTOP | MB_TOPMOST );
		}
		else
		{
			/* initialize eeehw */
			if ( eeehw_start ( ) )
			{
				WNDCLASSEX wc;
				MSG msg;
				/* register notification window class */
				mmemset ( &wc, 0, sizeof ( wc ) );
				wc.cbSize        = sizeof ( wc );
				wc.hInstance     = hInst;
				wc.lpfnWndProc   = NotifyWndProc;
				wc.lpszClassName = WNDSTRING;
				RegisterClassEx ( &wc );
				/* */
				mmemset ( &msg, 0, sizeof ( msg ) );
				do
				{
					/* create notification window */
					hMainWnd = CreateWindowEx ( 0, WNDSTRING, WNDSTRING, 0, 0, 0, 0, 0, NULL, NULL, hInst, hInst );
					if ( hMainWnd )
					{
						/* send cmdline */
						while ( st_height ( cmdline ) )
						{
							if ( 0 == mwcsicmp ( st_value_get ( cmdline, 0, 0 ), L"-e" ) ) SendMessage ( hMainWnd, WM_EXECMCMD, ( WPARAM ) st_value_get ( cmdline, 0, 1 ), 0 );
							st_row_del ( cmdline, 0 );
						}
						/* let it be */
						while ( GetMessage ( &msg, NULL, 0, 0 ) ) DispatchMessage ( &msg );
					}
				}
				while ( QUIT_RESTART == msg.wParam );
				ret = ( int ) msg.wParam;
				/* unregister class */
				UnregisterClass ( WNDSTRING, hInst );
				/* */
				eeehw_stop ( );
			}
		}
	}
	/* */
	return ret;
}

/* crt stub replacement */
int APIENTRY start ( )
{
	HINSTANCE hInst;
	wchar_t * lpCmdLine;
	int       nCmdShow;
	/* get hinstance */
	hInst = GetModuleHandle ( NULL );
	/* get cmdline */
	if ( lpCmdLine = GetCommandLine ( ) )
	{
		BOOL bQuoteOpened = FALSE;
		while ( 1 )
		{
			if ( ( 0 == * lpCmdLine ) || ( ( 0x20 >= * lpCmdLine ) && ( FALSE == bQuoteOpened ) ) ) break;
			if ( '"' == * lpCmdLine ) bQuoteOpened = ( FALSE == bQuoteOpened );
			++lpCmdLine;
		}
		while ( ( * lpCmdLine ) && ( 0x20 >= * lpCmdLine ) ) ++lpCmdLine;
	}
	else lpCmdLine = L"";
	/* get cmdshow */
	/* see GetStartupInfo to pass this explicitly */
	nCmdShow = SW_SHOWDEFAULT;
	/* ready */
	return wWinMain ( hInst, NULL, lpCmdLine, nCmdShow );
}
