/*
* eeectl
* Copyright (C) 2008 Anthony A Z <dci@cpp.in>
* http://www.cpp.in/dev/eeectl/
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License along
* with this program; if not, write to the Free Software Foundation, Inc.,
* 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#include <windows.h>
#include "runtime.h"
#include "dciio.h"
#include "eeehw.h"

/********************************************************************************/
/******************************** INTEL VIDEO ***********************************/
/********************************************************************************/
#define INTEL_VIDEO_BUS    0
#define INTEL_VIDEO_DEVICE 2

#define INTEL_VIDEO_LEGACY_BL 0xF4


/********************************************************************************/
/******************************** INTEL SMBUS ***********************************/
/********************************************************************************/
#define SMB_BASE 0x400
#define HSTS ( SMB_BASE + 0x00 ) /* host status */
#define HSTC ( SMB_BASE + 0x02 ) /* host control */
#define HCMD ( SMB_BASE + 0x03 ) /* host command */
#define HADR ( SMB_BASE + 0x04 ) /* transmit slave addr */
#define HDT0 ( SMB_BASE + 0x05 ) /* host data 0 */
#define HDT1 ( SMB_BASE + 0x06 ) /* host data 1 */
#define BLKD ( SMB_BASE + 0x07 ) /* host block data byte */
#define SPEC ( SMB_BASE + 0x08 ) /* packet error check */
#define SADR ( SMB_BASE + 0x09 ) /* recv slave addr */
#define SDT0 ( SMB_BASE + 0x0A ) /* recv slave data 0 */
#define SDT1 ( SMB_BASE + 0x0B ) /* recv slave data 1 */
#define AUXS ( SMB_BASE + 0x0C ) /* aux status */
#define AUXC ( SMB_BASE + 0x0D ) /* aux control */
#define SLPC ( SMB_BASE + 0x0E ) /* SMLink pin control */
#define SBPC ( SMB_BASE + 0x0F ) /* SMBus pin control */
#define SLVS ( SMB_BASE + 0x10 ) /* slave status */
#define SLVC ( SMB_BASE + 0x11 ) /* slave command */
#define NDAD ( SMB_BASE + 0x14 ) /* notify device addr */
#define NDLB ( SMB_BASE + 0x16 ) /* notify data low byte */
#define NDHB ( SMB_BASE + 0x17 ) /* notify data high byte */

/* host status port */
#define HSTS_HOST_BUSY ( 1 << 0 )
#define HSTS_INTR      ( 1 << 1 )
#define HSTS_DEV_ERR   ( 1 << 2 )
#define HSTS_BUS_ERR   ( 1 << 3 )
#define HSTS_FAILED    ( 1 << 4 )
#define HSTS_ALERT     ( 1 << 5 )
#define HSTS_INUSE     ( 1 << 6 )
#define HSTS_BYTE_DONE ( 1 << 7 )

#define HSTS_RESET     0xFF

/* host control port */
#define HSTC_INTREN    ( 1 << 0 )
#define HSTC_KILL      ( 1 << 1 )
#define HSTC_LAST_BYTE ( 1 << 5 )
#define HSTC_START     ( 1 << 6 )
#define HSTC_PEC_EN    ( 1 << 7 )

#define HSTC_CMD_SHIFT ( 2 )
#define HSTC_CMD_MASK  /* 0b111 */ 0x07

#define HSTC_CMD_QUICK /* 0b000 */ 0x00
#define HSTC_CMD_BYTE  /* 0b001 */ 0x01
#define HSTC_CMD_BDATA /* 0b010 */ 0x02
#define HSTC_CMD_WDATA /* 0b011 */ 0x03
#define HSTC_CMD_PCALL /* 0b100 */ 0x04
#define HSTC_CMD_BLOCK /* 0b101 */ 0x05
#define HSTC_CMD_I2CRD /* 0b110 */ 0x06
#define HSTC_CMD_BLPRC /* 0b111 */ 0x07

#define HSTC_CMD(a)    ( ( ( a ) &  HSTC_CMD_MASK  ) << HSTC_CMD_SHIFT )
#define HSTC_CMD_RD(a) ( ( ( a ) >> HSTC_CMD_SHIFT ) &  HSTC_CMD_MASK  )



/********************************************************************************/
/*************************** EMBEDDED CONTROLLER ********************************/
/********************************************************************************/
/* acpi-compiliant ports */
#define ACC_CMDP 0x0066
#define ACC_DATA 0x0062
#define ACC_OBF  0x01
#define ACC_IBF  0x02

/* index ports */
#define IDX_BASE 0x380
#define IDX_ADHI ( IDX_BASE + 0x01 )
#define IDX_ADLO ( IDX_BASE + 0x02 )
#define IDX_DATA ( IDX_BASE + 0x03 )

/* gpio regs */
#define GPIO_OF 0xFC00
#define GPIO_OE 0xFC10
#define GPIO_OD 0xFC20
#define GPIO_IS 0xFC30
#define GPIO_PU 0xFC40
#define GPIO_DE 0xFC50
#define GPIO_IE 0xFC60
#define GPIO_MC 0xFC70

/* kiwidrew's work */
#define EC_ST00				0xF451 // ACPI: Temperature of CPU (C)
#define EC_SC02				0xF463 // ACPI: Fan PWM duty cycle (%)
#define EC_SC05				0xF466 // ACPI: High byte of fan speed (RPM)
#define EC_SC06				0xF467 // ACPI: Low byte of fan speed (RPM)
#define EC_SFB3				0xF4D3 // ACPI: Flag byte containing SF25 (FANctrl)

#define GPIO_PIN_VOLTAGE	0x0066 //

#define EC_FPWMH1			0xFE26
#define EC_FPWML1			0xFE27


/* acpi-compiliant ports funcs */
WORD ec_acpireg_ibfx ( )
{
	WORD cycles = 0x1000;
	while ( cycles && ( dciIOPortByteRead ( ACC_CMDP ) & ACC_IBF ) )
	{
		//dciIOPortByteWrite ( 0x00E1, 0x00 );
		Sleep ( 10 );
		--cycles;
	}
	return cycles;
}

WORD ec_acpireg_obfx ( )
{
	WORD cycles = 0x1000;
	while ( cycles && ( 0 == ( dciIOPortByteRead ( ACC_CMDP ) & ACC_OBF ) ) )
	{
		//dciIOPortByteWrite ( 0x00E1, 0x00 );
		Sleep ( 10 );
		--cycles;
	}
	return cycles;
}

BYTE ec_acpireg_read ( BYTE reg )
{
	ec_acpireg_ibfx ( );
	dciIOPortByteWrite ( ACC_CMDP, 0x80 );
	ec_acpireg_ibfx ( );
	dciIOPortByteWrite ( ACC_DATA, reg );
	//ec_acpireg_obfx ( ); //not working
	ec_acpireg_ibfx ( );
	return dciIOPortByteRead ( ACC_DATA );
}

void ec_acpireg_write ( BYTE reg, BYTE value )
{
	ec_acpireg_ibfx ( );
	dciIOPortByteWrite ( ACC_CMDP, 0x81 );
	ec_acpireg_ibfx ( );
	dciIOPortByteWrite ( ACC_DATA, reg );
	ec_acpireg_ibfx ( );
	dciIOPortByteWrite ( ACC_DATA, value );
	ec_acpireg_ibfx ( );
}

/* index ports functions */
BYTE ec_idxreg_read ( WORD reg )
{
	dciIOPortByteWrite ( IDX_ADHI, ( reg >> 8 ) & 0xFF );
	dciIOPortByteWrite ( IDX_ADLO, ( reg >> 0 ) & 0xFF );
	return dciIOPortByteRead ( IDX_DATA );
}

void ec_idxreg_write ( WORD reg, BYTE value )
{
	dciIOPortByteWrite ( IDX_ADHI, ( reg >> 8 ) & 0xFF );
	dciIOPortByteWrite ( IDX_ADLO, ( reg >> 0 ) & 0xFF );
	dciIOPortByteWrite ( IDX_DATA, value );
}

/* gpio */
BYTE ec_gpio_pin_get ( WORD base, WORD pin )
{
	return ( ec_idxreg_read ( base + /* octet */ ( ( pin /* / 8 */ >> 3 ) & 0x1F ) ) & ( 1 << ( pin & /* 0b0111 */ 0x07 ) ) ) ? 1 : 0;
}

void ec_gpio_pin_set ( WORD base, WORD pin, BYTE val )
{
	WORD reg = base + /* octet */ ( ( pin /* / 8 */ >> 3 ) & 0x1F );
	WORD bit = 1 << ( pin & /* 0b0111 */ 0x07 );
	val = val ? ( ec_idxreg_read ( reg ) | bit ) : ( ec_idxreg_read ( reg ) & ~bit );
	ec_idxreg_write ( reg, val );
}

/* */
#define ec_gpio_output_function_get(pin) ec_gpio_pin_get(GPIO_OF,pin)
#define ec_gpio_output_function_set(pin,val) ec_gpio_pin_set(GPIO_OF,pin,val)
#define ec_gpio_output_enable_get(pin) ec_gpio_pin_get(GPIO_OE,pin)
#define ec_gpio_output_enable_set(pin,val) ec_gpio_pin_set(GPIO_OE,pin,val)
#define ec_gpio_output_data_get(pin) ec_gpio_pin_get(GPIO_OD,pin)
#define ec_gpio_output_data_set(pin,val) ec_gpio_pin_set(GPIO_OD,pin,val)

#if 0
BYTE ec_gpio_output_get ( WORD pin )
{
	return ( ec_idxreg_read ( GPIO_OD + /* octet */ ( ( pin /* / 8 */ >> 3 ) & 0x1F ) ) & ( 1 << ( pin & /* 0b0111 */ 0x07 ) ) ) ? 1 : 0;
}

void ec_gpio_output_set ( WORD pin, BYTE val )
{
	WORD reg = GPIO_OD + /* octet */ ( ( pin /* / 8 */ >> 3 ) & 0x1F );
	WORD bit = 1 << ( pin & /* 0b0111 */ 0x07 );
	val = val ? ( ec_idxreg_read ( reg ) | bit ) : ( ec_idxreg_read ( reg ) & ~bit );
	ec_idxreg_write ( reg, val );
}
#endif


/********************************************************************************/
/***************************** CLIENT FUNCTIONS *********************************/
/********************************************************************************/

/* start/stop */
BOOL eeehw_started = FALSE;
CRITICAL_SECTION cs;

unsigned char eeehw_start ( )
{
	if ( FALSE == eeehw_started )
	{
		if ( dciIOStart ( ) )
		{
			InitializeCriticalSection ( &cs );
			eeehw_started = TRUE;
			return 1;
		}
		else
		{
			wchar_t errbuf [ 128 ];
			wsprintf ( errbuf, L"Error %08X:%08X initializing IO driver.", dciIOErrorPoint ( ), dciIOErrorValue ( ) );
			MessageBox ( NULL, errbuf, L"dci", MB_ICONSTOP | MB_TOPMOST );
		}
	}
	else MessageBox ( NULL, L"EEE module already started.", L"dci", MB_ICONSTOP | MB_TOPMOST );
	/* */
	return 0;
}

unsigned char eeehw_stop ( )
{
	if ( eeehw_started )
	{
		dciIOStop ( );
		DeleteCriticalSection ( &cs );
		eeehw_started = FALSE;
		return 1;
	}
	return 0;
}

/* video */
unsigned char eeehw_video_bl_get ( )
{
	unsigned char ret;
	if ( dciIOPCIConfRead ( INTEL_VIDEO_BUS, INTEL_VIDEO_DEVICE, 0, INTEL_VIDEO_LEGACY_BL, sizeof ( ret ), &ret ) )
	{
		unsigned short conv = ( ( unsigned short ) ret ) * 100;
		if ( conv % 255 ) conv += 255;
		return ( unsigned char ) ( conv / 255 );
	}
	return 0;
}

unsigned char eeehw_video_bl_set ( unsigned char brightness )
{
	unsigned char ret = eeehw_video_bl_get ( );
	brightness = ( unsigned char ) ( ( ( ( unsigned short ) brightness ) * 255 ) / 100 );
	dciIOPCIConfWrite ( INTEL_VIDEO_BUS, INTEL_VIDEO_DEVICE, 0, INTEL_VIDEO_LEGACY_BL, sizeof ( brightness ), &brightness );
	return ret;
}

/* pll */
unsigned char eeehw_pll_read ( unsigned char offset, void * buffer, unsigned char max )
{
	unsigned char ret = 0;
	EnterCriticalSection ( &cs );
	/* loop */
	while ( ret < max )
	{
		unsigned char bsize;
		unsigned char i;
		/* request pll */
		dciIOPortByteWrite ( HCMD, offset + ret );
		Sleep ( 10 );
		dciIOPortByteWrite ( HADR, ( /* pll */ 0x69 << 1 ) | /* read */ 0x01 );
		Sleep ( 10 );
		dciIOPortByteWrite ( HSTS, HSTS_RESET );
		Sleep ( 10 );
		dciIOPortByteWrite ( HSTC, HSTC_CMD( HSTC_CMD_BLOCK ) | HSTC_START );
		/* wait */
		for ( i = 0; i < 255; ++i )
		{
			Sleep ( 10 );
			if ( dciIOPortByteRead ( HSTS ) & HSTS_INTR ) break;
		}
		/* finish request */
		bsize = dciIOPortByteRead ( HDT0 );
		dciIOPortByteRead ( HSTC );
		/* time to read pll data */
		if ( bsize && ( 255 != bsize ) )
		{
			for ( i = 0; i <= bsize; ++i )
			{
				unsigned char data = dciIOPortByteRead ( BLKD );
				if ( ( ret + i ) < max ) ( ( unsigned char * ) buffer ) [ ( ret + i ) ] = data;
			}
			ret += bsize;
		}
		else break;
	}
	/* */
	LeaveCriticalSection ( &cs );
	return ( ret < max ) ? ret : max;
}

unsigned char eeehw_pll_write ( unsigned char offset, void * buffer, unsigned char size )
{
	unsigned char i;
	EnterCriticalSection ( &cs );
	/* */
	dciIOPortByteRead ( HSTC );
	/* put data to the buffer */
	for ( i = 0; i < size; ++i )
	{
		dciIOPortByteWrite ( BLKD, ( ( BYTE * ) buffer ) [ i ] );
	}
	/* */
	dciIOPortByteRead ( HSTC );
	/* request */
	dciIOPortByteWrite ( HDT0, size );
	Sleep ( 10 );
	dciIOPortByteWrite ( HCMD, offset );
	Sleep ( 10 );
	dciIOPortByteWrite ( HADR, ( /* pll */ 0x69 << 1 ) | /* write */ 0x00 );
	Sleep ( 10 );
	dciIOPortByteWrite ( HSTS, HSTS_RESET );
	Sleep ( 10 );
	dciIOPortByteWrite ( HSTC, HSTC_CMD( HSTC_CMD_BLOCK ) | HSTC_START );
	/* wait */
	for ( i = 0; i < 255; ++i )
	{
		Sleep ( 10 );
		if ( dciIOPortByteRead ( HSTS ) & HSTS_INTR ) break;
	}
	/* */
	LeaveCriticalSection ( &cs );
	return size;
}


/* embedded controller */
unsigned char eeehw_voltage_get ( )
{
	unsigned char ret = 0;
	EnterCriticalSection ( &cs );
	ret = ec_gpio_output_data_get ( GPIO_PIN_VOLTAGE );
	LeaveCriticalSection ( &cs );
	return ret;
}

unsigned char eeehw_voltage_set ( unsigned char voltage )
{
	unsigned char ret = eeehw_voltage_get ( );
	EnterCriticalSection ( &cs );
	ec_gpio_output_data_set ( GPIO_PIN_VOLTAGE, voltage );
	LeaveCriticalSection ( &cs );
	return ret;
/*
	ec_acpireg_ibfx ( );
	dciIOPortByteWrite ( ACC_CMDP, 0xE1 );
	ec_acpireg_ibfx ( );
	dciIOPortByteWrite ( ACC_DATA, voltage ? 1 : 0 );
	ec_acpireg_ibfx ( );
*/
	return 0;
}

unsigned char eeehw_fan_mode_get ( )
{
	unsigned char reg = 0;
	EnterCriticalSection ( &cs );
	reg = ec_idxreg_read ( EC_SFB3 );
	LeaveCriticalSection ( &cs );
	return ( reg & /* */ 0x02 ) ? FAN_MODE_MANUAL : FAN_MODE_AUTO;
}

unsigned char eeehw_fan_mode_set ( unsigned char mode )
{
	unsigned char sfb3 = 0;
	EnterCriticalSection ( &cs );
	sfb3 = ec_idxreg_read ( EC_SFB3 );
	ec_idxreg_write ( EC_SFB3, ( FAN_MODE_MANUAL == mode ) ? ( sfb3 | 0x02 ) : ( sfb3 & ~0x02 ) );
	LeaveCriticalSection ( &cs );
	return ( sfb3 & 0x02 ) ? FAN_MODE_MANUAL : FAN_MODE_AUTO;
}

unsigned char eeehw_fan_speed_get ( )
{
	unsigned char ret = 0;
//	unsigned short pwm;
	EnterCriticalSection ( &cs );

	ret = ec_idxreg_read ( EC_SC02 );



//	pwm = ( ( ( unsigned short ) ec_idxreg_read ( EC_FPWMH1 ) ) << 8 ) | ec_idxreg_read ( EC_FPWML1 );
//	if ( 0x01CA <= pwm ) ret = 100;
//	else ret = ( pwm * 100 ) / 0x1CA;
	/* */
	LeaveCriticalSection ( &cs );
	return ret;
}	

unsigned char eeehw_fan_speed_set ( unsigned char speed )
{
	unsigned char ret = eeehw_fan_speed_get ( );
//	unsigned short pwm;

//	wchar_t b[64];
	EnterCriticalSection ( &cs );

	ec_idxreg_write ( EC_SC02, ( speed > 100 ) ? 100 : speed );

//	if ( 100 <= speed ) pwm = 0x1CA; //0x0FFF;
//	else pwm = ( ( ( unsigned short ) speed ) * 0x1CA ) / 100;
//	ec_idxreg_write ( EC_FPWMH1, pwm >> 8 );
//	Sleep ( 100 );
//	ec_idxreg_write ( EC_FPWML1, pwm & 0xFF );

//	ec_gpio_output_data_set ( 18, 0 );
//	ec_gpio_output_function_set ( 18, 1 );
//	ec_gpio_output_enable_set ( 18, 0 );

//	wsprintf ( b, L"%d, %d: %04X=%02X %04X=%02X", ret, speed, EC_FPWMH1, pwm >> 8, EC_FPWML1, pwm & 0xFF );
//	MessageBox ( NULL, b, NULL, MB_TOPMOST );
	/* */
	LeaveCriticalSection ( &cs );
	return ret;
}

unsigned char eeehw_temp_get ( )
{
	unsigned char ret = 0;
	EnterCriticalSection ( &cs );
	ret = ec_idxreg_read ( EC_ST00 );
	LeaveCriticalSection ( &cs );
	return ret;
}
