/*
* eeectl
* Copyright (C) 2008 Anthony A Z <dci@cpp.in>
* http://www.cpp.in/dev/eeectl/
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License along
* with this program; if not, write to the Free Software Foundation, Inc.,
* 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#include <windows.h>
#include "runtime.h"
#include "table.h"
#include "eeehw.h"
#include "ss_fan.h"

#define CFLEN 2048


#define COL_SPEED 0
#define COL_ON    1
#define COL_OFF   2
#define COL_NUM   3


#define FAN_MMODE_NATIVE 0
#define FAN_MMODE_AUTO   1
#define FAN_MMODE_MANUAL 2



eeectl_loader_t * gcore;

UINT uMode;
UINT uSpeed;

sth_t tblProfiles;

BOOL bSuspended;
UINT uPreSuspendSpeed;

HBITMAP hMenuSelect = NULL;

BOOL fan_init ( eeectl_loader_t * core )
{
	UINT i;
	wchar_t * confstring = ( wchar_t * ) mmalloc ( CFLEN * sizeof ( wchar_t ) );
	gcore = core;
	/* load profiles */
	gcore->setup_read ( &eeectl_subsys_fan, L"Profiles", confstring, CFLEN );
	tblProfiles = st_create ( COL_NUM );
	if ( st_serial_read ( tblProfiles, confstring ) )
	{
		for ( i = 0; i < st_height ( tblProfiles ); ++i )
		{
			UINT j;
			for ( j = i + 1; j < st_height ( tblProfiles ); ++j )
			{
				if ( mwtoi ( st_value_get ( tblProfiles, i, COL_SPEED ) ) == mwtoi ( st_value_get ( tblProfiles, j, COL_SPEED ) ) )
				{
					gcore->pass_msg ( &eeectl_subsys_fan, MSG_ERROR, L"Duplicate value: '%s'", st_value_get ( tblProfiles, j, COL_SPEED ) );
					st_row_del ( tblProfiles, j );
					--j;
				}
			}
		}
	}
	/* load state */
	gcore->state_read ( &eeectl_subsys_fan, L"Mode", confstring, CFLEN );
	uMode = mwtoi ( confstring );
	if ( FAN_MMODE_NATIVE == uMode )
	{
		eeehw_fan_mode_set ( FAN_MODE_AUTO );
	}
	else
	{
		eeehw_fan_mode_set ( FAN_MODE_MANUAL );
	}
	/* */
	if ( FAN_MMODE_MANUAL == uMode )
	{
		gcore->state_read ( &eeectl_subsys_fan, L"Speed", confstring, CFLEN );
		uSpeed = mwtoi ( confstring );
		eeehw_fan_speed_set ( uSpeed );
	}
	else uSpeed = eeehw_fan_speed_get ( );
	/* create bitmap for menu */
	if ( NULL == hMenuSelect )
	{
		UINT   i, j;
		UINT   cxmc = GetSystemMetrics ( SM_CXMENUCHECK );
		UINT   cymc = GetSystemMetrics ( SM_CYMENUCHECK );
		UINT   line = 2 * ( cxmc % 16 ? ( ( cxmc + 16 ) / 16 ) : ( cxmc / 16 ) );
		BYTE * buff = ( BYTE * ) mmalloc ( line * cymc );
		/* fill with white/transparent */
		mmemset ( buff, 0xFF, line * cymc );
		/* draw black triangle */
		for ( i = cymc / 6; i < cymc - ( cymc / 6 ); ++i )
		{
			UINT dots = ( i <= ( cymc / 2 ) ) ? ( i - ( cymc / 6 ) + 1 ) : ( ( cymc - ( cymc / 6 ) ) - i );
			for ( j = cxmc / 3; j < ( ( cxmc / 3 ) + dots ); ++j )
			{
				buff [ ( i * line ) + ( j / 8 ) ] &= ~( 1 << ( 7 - ( j % 8 ) ) );
			}
		}
		/* create bitmap */
		hMenuSelect = CreateBitmap ( cxmc, cymc, 1, 1, buff );
		/* */
		mfree ( buff );
	}
	/* */
	return TRUE;
}

void fan_save ( )
{
	wchar_t b [ 64 ];
	wsprintf ( b, L"%d", uMode );
	gcore->state_write ( &eeectl_subsys_fan, L"Mode", b );
	wsprintf ( b, ( FAN_MMODE_MANUAL == uMode ) ? L"%d" : L"", uSpeed );
	gcore->state_write ( &eeectl_subsys_fan, L"Speed", b );
}

void fan_stop ( BOOL restart )
{
	if ( FALSE == restart )
	{
		eeehw_fan_mode_set ( FAN_MODE_AUTO );
		DeleteObject ( hMenuSelect );
		hMenuSelect = NULL;
	}
}

void fan_suspend ( )
{
	if ( FAN_MMODE_NATIVE != uMode )
	{
		uPreSuspendSpeed = uSpeed;
		bSuspended = TRUE;
	}
}

void fan_resume ( )
{
	if ( bSuspended )
	{
		uSpeed = uPreSuspendSpeed;
		eeehw_fan_mode_set ( FAN_MODE_MANUAL );
		eeehw_fan_speed_set ( ( BYTE ) uSpeed );
		bSuspended = FALSE;
	}
}

UINT fan_icontxt ( wchar_t * icon, wchar_t * txt, size_t max )
{
	return wsprintf ( txt, /* ';' means '%' here */ L"%d;", uSpeed );
}

UINT fan_menucrt ( HMENU menu, UINT base )
{
	UINT ret = 0;
	UINT i;
	/* */
	BOOL bMatch = FALSE;
	AppendMenu ( menu, MF_BYCOMMAND | MF_STRING | ( ( FAN_MMODE_NATIVE == uMode ) ? MF_CHECKED : 0 ), base + ret, L"Native" );
	++ret;
	AppendMenu ( menu, MF_BYCOMMAND | MF_STRING | ( ( FAN_MMODE_AUTO == uMode ) ? MF_CHECKED : 0 ), base + ret, L"Automatic" );
	++ret;
	/* */
	for ( i = 0; i < st_height ( tblProfiles ); ++i )
	{
		UINT    speed = mwtoi ( st_value_get ( tblProfiles, i, COL_SPEED ) );
		wchar_t menucap [ 128 ];
		wsprintf ( menucap, L"%d%%", speed );

		if ( FALSE == bMatch ) bMatch = ( speed == uSpeed );
		AppendMenu ( menu, MF_BYCOMMAND | MF_STRING | ( ( speed == uSpeed ) ? MF_CHECKED : 0 ), base + ret, menucap );
		if ( FAN_MMODE_MANUAL != uMode ) SetMenuItemBitmaps ( menu, base + ret, MF_BYCOMMAND, NULL, hMenuSelect );
		++ret;
	}
	/* non-listed? */
	if ( FALSE == bMatch )
	{
		wchar_t menucap [ 64 ];
		wsprintf ( menucap, L"Custom (%d%%)", uSpeed );
		AppendMenu ( menu, MF_BYCOMMAND | MF_STRING | MF_CHECKED | MF_GRAYED, base + ret, menucap );
		if ( FAN_MMODE_MANUAL != uMode ) SetMenuItemBitmaps ( menu, base + ret, MF_BYCOMMAND, NULL, hMenuSelect );
		++ret;
	}
	/* */
	return ret;
}

BOOL fan_menuclk ( UINT id )
{
	if ( 0 == id )
	{
		uMode = FAN_MMODE_NATIVE;
		eeehw_fan_mode_set ( FAN_MODE_AUTO );
	}
	else if ( 1 == id )
	{
		uMode    = FAN_MMODE_AUTO;
		eeehw_fan_mode_set ( FAN_MODE_MANUAL );
	}
	else
	{
		uMode    = FAN_MMODE_MANUAL;
		uSpeed   = mwtoi ( st_value_get ( tblProfiles, id - 2, COL_SPEED ) );
		eeehw_fan_mode_set ( FAN_MODE_MANUAL );
		eeehw_fan_speed_set ( uSpeed );
	}
	gcore->ntf_icon_upd ( &eeectl_subsys_fan, NULL );
	return TRUE;
}

BOOL fan_ntfevnt ( wchar_t * evname )
{
	if ( 0 == mwcsicmp ( evname, L"_native" ) )
	{
		uMode = FAN_MMODE_NATIVE;
		eeehw_fan_mode_set ( FAN_MODE_AUTO );
		gcore->ntf_icon_upd ( &eeectl_subsys_fan, NULL );
		return TRUE;
	}
	else if ( 0 == mwcsicmp ( evname, L"_auto" ) )
	{
		uMode    = FAN_MMODE_AUTO;
		eeehw_fan_mode_set ( FAN_MODE_MANUAL );
		gcore->ntf_icon_upd ( &eeectl_subsys_fan, NULL );
		return TRUE;
	}
	else if ( IS_NUMERIC ( * evname ) )
	{
		uMode  = FAN_MMODE_MANUAL;
		uSpeed = mwtoi ( evname );
		eeehw_fan_mode_set ( FAN_MODE_MANUAL );
		eeehw_fan_speed_set ( uSpeed );
		gcore->ntf_icon_upd ( &eeectl_subsys_fan, NULL );
		return TRUE;
	}
	return FALSE;
}

void fan_ntftime ( )
{
	if ( FAN_MMODE_NATIVE == uMode )
	{
		unsigned char nSpeed = eeehw_fan_speed_get ( );
		if ( nSpeed != uSpeed )
		{
			uSpeed = nSpeed;
			gcore->ntf_icon_upd ( &eeectl_subsys_fan, NULL );
		}
	}
	else if ( FAN_MMODE_AUTO == uMode )
	{
		UINT i;
		UINT uCurrent = 0;
		UINT uSugg = 0;
		UINT uTemp = eeehw_temp_get ( );
		/* sync */
		uSpeed = eeehw_fan_speed_get ( );
		/* try to find current profile */
		for ( i = 0; i < st_height ( tblProfiles ); ++i )
		{
			if ( ( uSpeed == mwtoi ( st_value_get ( tblProfiles, i, COL_SPEED ) ) ) && mwtoi ( st_value_get ( tblProfiles, i, COL_ON ) ) )
			{
				uCurrent = i + 1;
				break;
			}
		}
		/* try to find most suitable profile */
		for ( i = 0; i < st_height ( tblProfiles ); ++i )
		{
			UINT con = ( UINT ) mwtoi ( st_value_get ( tblProfiles, i, COL_ON ) );
			if ( con && ( uTemp >= con ) && ( ( 0 == uSugg ) || ( con > ( UINT ) mwtoi ( st_value_get ( tblProfiles, uSugg - 1, COL_ON ) ) ) ) ) uSugg = i + 1;
		}
		/* no profile to select? */
		if ( ( 0 == uSugg ) && ( ( 0 == uCurrent ) || ( uTemp <= ( UINT ) mwtoi ( st_value_get ( tblProfiles, uCurrent - 1, COL_OFF ) ) ) ) )
		{
			uCurrent = 0;
			if ( 0 != uSpeed )
			{
				uSpeed   = 0;
				eeehw_fan_speed_set ( uSpeed );
				gcore->ntf_icon_upd ( &eeectl_subsys_fan, NULL );
			}
		}
		/* validate change */
		else if ( uSugg && (
			/* no profile selected */ ( 0 == uCurrent ) ||
			/* most suitable is higher */ ( mwtoi ( st_value_get ( tblProfiles, uSugg - 1, COL_ON ) ) > mwtoi ( st_value_get ( tblProfiles, uCurrent - 1, COL_ON ) ) ) ||
			/* OFF temp is reached */ ( uTemp <= ( UINT ) mwtoi ( st_value_get ( tblProfiles, uCurrent - 1, COL_OFF ) ) )
			) )
		{
			uCurrent = uSugg;
			uSpeed = mwtoi ( st_value_get ( tblProfiles, uCurrent - 1, COL_SPEED ) );
			eeehw_fan_speed_set ( uSpeed );
			gcore->ntf_icon_upd ( &eeectl_subsys_fan, NULL );
		}
	}
}

eeectl_subsys_t eeectl_subsys_fan = { L"Fan", L"eeectl fan control module", fan_init, fan_save, fan_stop, fan_suspend, fan_resume, fan_icontxt, fan_menucrt, fan_menuclk, fan_ntfevnt, fan_ntftime };
