<?php
header('Content-Type: application/json; charset=utf-8');

// Path to the json file (adjust path if needed)
define('LIKE_FILE', __DIR__ . '/like.json');

// Helper: get client IP (best-effort)
function client_ip() {
    // Basic checks; depending on hosting you might want to use only REMOTE_ADDR
    if (!empty($_SERVER['HTTP_CLIENT_IP'])) return $_SERVER['HTTP_CLIENT_IP'];
    if (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
        $ips = explode(',', $_SERVER['HTTP_X_FORWARDED_FOR']);
        return trim($ips[0]);
    }
    return $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
}

// Read JSON safely with shared lock
function read_data() {
    $file = LIKE_FILE;
    if (!file_exists($file)) {
        file_put_contents($file, "{}");
    }
    $fp = fopen($file, 'r');
    if (!$fp) return [];
    flock($fp, LOCK_SH);
    $json = stream_get_contents($fp);
    flock($fp, LOCK_UN);
    fclose($fp);
    $data = json_decode($json, true);
    return is_array($data) ? $data : [];
}

// Write JSON safely with exclusive lock
function write_data($data) {
    $file = LIKE_FILE;
    $fp = fopen($file, 'c+');
    if (!$fp) return false;
    flock($fp, LOCK_EX);
    ftruncate($fp, 0);
    rewind($fp);
    $written = fwrite($fp, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));
    fflush($fp);
    flock($fp, LOCK_UN);
    fclose($fp);
    return $written !== false;
}

// Get inputs
$action = $_REQUEST['action'] ?? 'get';
$postid = $_REQUEST['postid'] ?? '';
// Normalize post key
$postkey = $postid === '' ? '' : 'post_' . preg_replace('/[^a-zA-Z0-9_\-]/', '', $postid);
$ip = client_ip();

if ($postkey === '') {
    echo json_encode(['ok' => false, 'error' => 'postid required']);
    exit;
}

$data = read_data();

if (!isset($data[$postkey])) {
    $data[$postkey] = ['likes' => 0, 'ips' => []];
}

if ($action === 'get') {
    $liked = in_array($ip, $data[$postkey]['ips'], true);
    echo json_encode([
        'ok' => true,
        'postid' => $postid,
        'likes' => (int)$data[$postkey]['likes'],
        'liked' => $liked
    ]);
    exit;
}

if ($action === 'toggle') {
    // toggle like for this IP
    $ips = $data[$postkey]['ips'];
    if (!is_array($ips)) $ips = [];
    $idx = array_search($ip, $ips, true);
    if ($idx === false) {
        // add like
        $ips[] = $ip;
        $data[$postkey]['likes'] = max(0, (int)$data[$postkey]['likes']) + 1;
        $liked = true;
    } else {
        // remove like
        array_splice($ips, $idx, 1);
        $data[$postkey]['likes'] = max(0, (int)$data[$postkey]['likes'] - 1);
        $liked = false;
    }
    // keep unique IPs
    $data[$postkey]['ips'] = array_values(array_unique($ips));

    $ok = write_data($data);
    if (!$ok) {
        echo json_encode(['ok' => false, 'error' => 'failed to write file']);
        exit;
    }

    echo json_encode([
        'ok' => true,
        'postid' => $postid,
        'likes' => (int)$data[$postkey]['likes'],
        'liked' => $liked
    ]);
    exit;
}

echo json_encode(['ok' => false, 'error' => 'unknown action']);

