<?php
// mobilefirst.php — shared config, storage + helpers (no MySQL).
// Minimal, mobile-first forum using data.json as storage.
// ------------------------------------------------------------------
declare(strict_types=1);

// set lifetime to 7 days (seconds)
$lifetime = 7 * 24 * 60 * 60; // 604800

// Set cookie params BEFORE session_start()
// path, domain, secure, httponly, samesite (PHP 7.3+ supports samesite via array)
session_set_cookie_params([
    'lifetime' => $lifetime,
    'path' => '/',
    'domain' => '',           // set to your domain if needed
    'secure' => isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off', // require HTTPS
    'httponly' => true,
    'samesite' => 'Lax'       // or 'Strict'/'None' (if using cross-site)
]);

// Ensure PHP's session GC keeps session data for the same duration
ini_set('session.gc_maxlifetime', (string)$lifetime);

// Optionally choose a session save path you control:
session_save_path(__DIR__ . '/sessions'); // make sure writable and secure

session_start();

// Path to data file
define('DATA_FILE', __DIR__ . '/data.json');

// Admin password (change this!)
define('ADMIN_PASSWORD', '12345678'); // Set a strong password

// Online timeout (seconds)
define('ONLINE_TTL', 300);

// Ensure data file exists
if (!file_exists(DATA_FILE)) {
    $seed = [
        'meta' => ['created_at' => time(), 'version' => '1.0.0'],
        'users' => [], // user_id => ['username' => 'Guest123', 'created_at'=>ts]
        'topics' => [], // topic_id => {id,title,author_id,author_ip,created_at,updated_at,closed,sticky}
        'posts'  => [], // post_id => {id,topic_id,content,author_id,author_ip,created_at,updated_at}
        'stats'  => [
            'hits' => [],         // 'YYYY-MM-DD' => int
            'visitors' => [],     // 'YYYY-MM-DD' => ['ip1'=>1, 'ip2'=>1, ...]
            'online' => [],       // 'ip' => last_seen_ts
            'blocked_ips' => []   // list of IPs
        ],
        'seq' => ['topic'=>1, 'post'=>1, 'user'=>1]
    ];
    file_put_contents(DATA_FILE, json_encode($seed, JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE));
}

// ------- Basic IO -------
function load_data() {
    $json = file_get_contents(DATA_FILE);
    $data = json_decode($json, true);
    if (!is_array($data)) { $data = []; }
    return $data;
}

function save_data($data) {
    file_put_contents(DATA_FILE, json_encode($data, JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE));
}

// ------- Utilities -------
function ip(): string {
    return $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
}

function is_blocked_ip($data, $ip): bool {
    return in_array($ip, $data['stats']['blocked_ips'] ?? [], true);
}

function require_not_blocked() {
    $d = load_data();
    if (is_blocked_ip($d, ip())) {
        http_response_code(403);
        include "header.php";
        echo "<div class='card'>Your IP is blocked.</div>";
        include "footer.php";
        exit;
    }
}

function h($s) { return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }

function now() { return time(); }

function today_date(): string { return date('Y-m-d'); }

function record_hit_and_online() {
    $d = load_data();
    $t = today_date();
    // hits
    if (!isset($d['stats']['hits'][$t])) $d['stats']['hits'][$t] = 0;
    $d['stats']['hits'][$t]++;
    // visitors
    if (!isset($d['stats']['visitors'][$t])) $d['stats']['visitors'][$t] = [];
    $d['stats']['visitors'][$t][ip()] = 1;
    // online
    $d['stats']['online'][ip()] = now();
    // clean stale
    foreach ($d['stats']['online'] as $pip => $ts) {
        if (now() - (int)$ts > ONLINE_TTL) unset($d['stats']['online'][$pip]);
    }
    save_data($d);
}

function online_count(): int {
    $d = load_data();
    $count = 0;
    foreach ($d['stats']['online'] as $pip => $ts) {
        if (now() - (int)$ts <= ONLINE_TTL) $count++;
    }
    return $count;
}

// ------- Sessions & Users -------
function ensure_user() {
    $d = load_data();
    if (!isset($_SESSION['user_id'])) {
        $uid = $d['seq']['user']++;
        $username = 'Guest' . $uid;
        $d['users'][(string)$uid] = ['username'=>$username, 'created_at'=>now()];
        $_SESSION['user_id'] = (string)$uid;
        save_data($d);
    }
    return $_SESSION['user_id'];
}

function current_user() {
    $d = load_data();
    $uid = ensure_user();
    return ['id'=>$uid, 'username'=>$d['users'][(string)$uid]['username'] ?? ('Guest'.$uid)];
}

function is_admin(): bool {
    return isset($_SESSION['is_admin']) && $_SESSION['is_admin'] === true;
}

// ------- Auth actions -------
function admin_login($password) {
    if ($password === ADMIN_PASSWORD) {
        $_SESSION['is_admin'] = true;
        return true;
    }
    return false;
}

function admin_logout() { unset($_SESSION['is_admin']); }

// ------- ID seq helpers -------
function next_topic_id(&$d) {
    $id = $d['seq']['topic']++;
    return (string)$id;
}

function next_post_id(&$d) {
    $id = $d['seq']['post']++;
    return (string)$id;
}

// ------- Include BBCode/Emoji parsers -------
require_once __DIR__ . '/bbcodes.php';
require_once __DIR__ . '/emojis.php';

// ------- Content rendering -------
function render_content($text) {
    $text = trim((string)$text);
    $text = substr($text, 0, 5000); // hard limit
    $text = parse_bbcodes($text);
    $text = parse_emojis($text);
    return $text;
}

// Call tracking at the start of each request
require_not_blocked();
record_hit_and_online();

?>
